<?php
// qbo_token_manager.php
// Handles automatic QBO access token refresh + persistent storage (file-based).

function qbo_token_store_path(array $config): string {
  return $config['qbo']['token_store_file'] ?? (__DIR__ . '/qbo_tokens.json');
}

function qbo_token_store_load(array $config): array {
  $file = qbo_token_store_path($config);
  if (!file_exists($file)) return [];
  $raw = file_get_contents($file);
  $data = json_decode((string)$raw, true);
  return is_array($data) ? $data : [];
}

function qbo_token_store_save(array $config, array $tokenData): void {
  $file = qbo_token_store_path($config);

  $existing = qbo_token_store_load($config);

  // Refresh token may rotate. If API doesn't return a new one, keep existing/current config.
  $refresh = $tokenData['refresh_token']
    ?? ($existing['refresh_token'] ?? ($config['qbo']['refresh_token'] ?? null));

  $out = [
    'access_token'  => $tokenData['access_token'] ?? ($existing['access_token'] ?? null),
    'refresh_token' => $refresh,
    'expires_in'    => $tokenData['expires_in'] ?? ($existing['expires_in'] ?? null),
    'created_at'    => time(),             // when we received this access token
    'saved_at'      => date('c'),
  ];

  file_put_contents($file, json_encode($out, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
}

function qbo_token_is_expired(array $config, array $store): bool {
  if (empty($store['access_token'])) return true;
  if (empty($store['created_at']) || empty($store['expires_in'])) return true;

  $leeway = (int)($config['qbo']['token_refresh_leeway_sec'] ?? 180); // refresh early (3 min default)
  $expiresAt = (int)$store['created_at'] + (int)$store['expires_in'];

  return (time() + $leeway) >= $expiresAt;
}

/**
 * Always returns a valid access token or throws.
 * Requires qbo_refresh_access_token($config) to exist (in your existing QBO file).
 */
function qbo_get_valid_access_token(array &$config): string {
  $store = qbo_token_store_load($config);

  // prefer stored refresh token if present
  if (!empty($store['refresh_token'])) {
    $config['qbo']['refresh_token'] = $store['refresh_token'];
  }

  if (!$store || qbo_token_is_expired($config, $store)) {
    log_line($config, "QBO token missing/expired — refreshing...");

    $tokenData = qbo_refresh_access_token($config);

    // IMPORTANT: persist rotated refresh_token if returned
    qbo_token_store_save($config, $tokenData);

    $store = qbo_token_store_load($config);

    if (empty($store['access_token'])) {
      throw new Exception("QBO refresh succeeded but access_token missing in token store");
    }

    log_line($config, "QBO token refreshed + stored");
  }

  return (string)$store['access_token'];
}
