<?php
// qbo_call_wrappers.php
// Wraps qbo_query/qbo_post so they auto-refresh and retry once on 401 token expired.

function qbo_should_retry_token_expired(Exception $e): bool {
  $m = $e->getMessage();
  return (str_contains($m, 'HTTP 401') && str_contains($m, 'Token expired'))
      || (str_contains($m, '"statusCode":401') && str_contains($m, 'Token expired'))
      || (str_contains($m, 'AuthenticationFailed') && str_contains($m, 'Token expired'));
}

/**
 * Call a QBO function that takes ($accessToken) and returns something.
 */
function qbo_run_with_auto_refresh(array &$config, callable $fn) {
  $accessToken = qbo_get_valid_access_token($config);

  try {
    return $fn($accessToken);
  } catch (Exception $e) {
    if ($e instanceof Exception && qbo_should_retry_token_expired($e)) {
      log_line($config, "QBO 401 Token expired — refreshing and retrying once...");
      // Force refresh by wiping stored access token (keep refresh token)
      $store = qbo_token_store_load($config);
      if (is_array($store)) {
        $store['access_token'] = null;
        $store['created_at'] = 0;
        $store['expires_in'] = 0;
        file_put_contents(
          $config['qbo']['token_store_file'] ?? (__DIR__ . '/qbo_tokens.json'),
          json_encode($store, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)
        );
      }
      $accessToken = qbo_get_valid_access_token($config);
      return $fn($accessToken);
    }
    throw $e;
  }
}
