<?php
// lib/util/log.php

function ensure_dir(string $dir): void {
  if (!is_dir($dir)) mkdir($dir, 0775, true);
}

function log_line(array $config, string $msg): void {
  $line = "[" . gmdate('c') . "] " . $msg . "\n";
  echo $line;

  $logDir = $config['log_dir'] ?? (__DIR__ . '/../../logs');
  ensure_dir($logDir);

  file_put_contents($logDir . "/process_queue.log", $line, FILE_APPEND);
}

function fail(array $config, string $msg): void {
  log_line($config, "FATAL: " . $msg);
  exit(1);
}

function indent_lines(string $text, int $spaces): string {
  $pad = str_repeat(' ', $spaces);
  $text = rtrim($text);
  if ($text === '') return $pad . "(none)";
  return $pad . str_replace("\n", "\n{$pad}", $text);
}

function pretty_body($body): string {
  if ($body === null || $body === '') return "(empty)";

  $trim = ltrim((string)$body);
  if ($trim !== '' && ($trim[0] === '{' || $trim[0] === '[')) {
    $decoded = json_decode((string)$body, true);
    if (json_last_error() === JSON_ERROR_NONE) {
      return json_encode($decoded, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    }
  }
  return (string)$body;
}

/**
 * Write a detailed HTTP log file into /logs like your old monolith did.
 * Expects $resp in the same shape returned by http_request().
 */
function log_http_file(array $config, string $tag, array $resp): void {
  $logDir = $config['log_dir'] ?? (__DIR__ . '/../../logs');
  ensure_dir($logDir);

  $ts = gmdate('Y-m-d_His');
  $status = $resp['status'] ?? 0;
  $fn = $logDir . "/{$ts}_{$tag}_{$status}.log";

  $req = $resp['request'] ?? ['method'=>'?','url'=>'?','headers'=>[],'body'=>null];

  $out =
    "[" . gmdate('c') . "] {$tag}\n\n" .
    "REQUEST:\n  {$req['method']} {$req['url']}\n  Headers:\n" .
      indent_lines(is_array($req['headers'] ?? null) ? implode("\n", $req['headers']) : (string)($req['headers'] ?? ''), 4) . "\n" .
    "  Body:\n" . indent_lines(pretty_body($req['body'] ?? null), 4) . "\n\n" .
    "RESPONSE:\n  Status: {$status}\n  Headers:\n" .
      indent_lines(trim((string)($resp['headers'] ?? '')), 4) . "\n" .
    "  Body:\n" . indent_lines(pretty_body($resp['body'] ?? ''), 4) . "\n";

  file_put_contents($fn, $out);
}
