<?php
// lib/util/http.php

/**
 * Low-level HTTP helper (cURL).
 * Returns: [
 *   'status' => int,
 *   'headers' => string,
 *   'body' => string,
 *   'request' => ['method'=>..., 'url'=>..., 'headers'=>array, 'body'=>string|null]
 * ]
 */
function http_request(string $method, string $url, array $headers = [], ?string $body = null): array {
  $ch = curl_init($url);

  curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
  curl_setopt($ch, CURLOPT_HEADER, true);
  curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
  curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
  curl_setopt($ch, CURLOPT_TIMEOUT, 60);

  if ($body !== null) {
    curl_setopt($ch, CURLOPT_POSTFIELDS, $body);
  }

  $raw = curl_exec($ch);

  if ($raw === false) {
    $err = curl_error($ch);
    curl_close($ch);
    throw new Exception("cURL failed: {$err}");
  }

  $status = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  $headerSize = (int)curl_getinfo($ch, CURLINFO_HEADER_SIZE);

  $rawHeaders = substr($raw, 0, $headerSize);
  $respBody   = substr($raw, $headerSize);

  curl_close($ch);

  return [
    'status' => $status,
    'headers' => $rawHeaders,
    'body' => $respBody,
    'request' => [
      'method' => $method,
      'url' => $url,
      'headers' => $headers,
      'body' => $body,
    ],
  ];
}
