<?php
// lib/sources/net32/normalize.php

/**
 * Pick the best available license record from NET32 payload.
 * Prefer customerOrderLicense (order-specific), then fallback to customerLicense.
 */
function net32_pick_first_license(array $o): ?array {
  $candidates = [];

  if (!empty($o['customerOrderLicense']) && is_array($o['customerOrderLicense'])) {
    $candidates = $o['customerOrderLicense'];
  } elseif (!empty($o['customerLicense']) && is_array($o['customerLicense'])) {
    $candidates = $o['customerLicense'];
  }

  foreach ($candidates as $lic) {
    if (!is_array($lic)) continue;

    $num  = trim((string)($lic['licenseNumber'] ?? ''));
    $type = trim((string)($lic['licenseType'] ?? ''));
    $ag   = trim((string)($lic['licenseAgency'] ?? ''));
    $exp  = trim((string)($lic['expirationDate'] ?? ''));

    // require at least something meaningful
    if ($num === '' && $type === '' && $ag === '' && $exp === '') continue;

    return [
      'license_number'   => $num,
      'license_type'     => $type,
      'license_agency'   => $ag,
      'expiration_date'  => $exp,
    ];
  }

  return null;
}

function net32_normalize(array $o): array {
  $externalOrderId = (string)($o['orderId'] ?? '');
  if ($externalOrderId === '') throw new Exception("NET32 payload missing orderId");

  $customerName = trim((string)($o['customerName'] ?? ''));
  $ship = $o['shippingAddress'] ?? [];
  $bill = $o['billingAddress'] ?? [];

  $shipTo = [
    'name' => trim((string)($ship['name'] ?? $customerName)),
    'phone' => trim((string)($ship['phone'] ?? '')),
    'streets' => is_array($ship['streets'] ?? null) ? $ship['streets'] : [],
    'city' => (string)($ship['city'] ?? ''),
    'region' => (string)($ship['region'] ?? ''),
    'postalCode' => (string)($ship['postalCode'] ?? ''),
    'country' => (string)($ship['country'] ?? ''),
  ];

  $billTo = [
    'name' => trim((string)($bill['name'] ?? $customerName)),
    'phone' => trim((string)($bill['phone'] ?? '')),
    'streets' => is_array($bill['streets'] ?? null) ? $bill['streets'] : [],
    'city' => (string)($bill['city'] ?? ''),
    'region' => (string)($bill['region'] ?? ''),
    'postalCode' => (string)($bill['postalCode'] ?? ''),
    'country' => (string)($bill['country'] ?? ''),
  ];

  $shippingSubtotal = money((float)(to_float($o['shippingSubtotal'] ?? 0) ?? 0.0));
  $estimatedTax     = money((float)(to_float($o['estimatedTax'] ?? 0) ?? 0.0));
  $orderTotal       = money((float)(to_float($o['orderTotal'] ?? 0) ?? 0.0));

  $lines = [];
  $lineItems = $o['lineItems'] ?? [];
  if (is_array($lineItems)) {
    foreach ($lineItems as $li) {
      if (!is_array($li)) continue;

      $qty = (float)(to_float($li['quantity'] ?? 1) ?? 1.0);
      if ($qty <= 0) $qty = 1.0;

      $unit = money((float)(to_float($li['unitPrice'] ?? 0) ?? 0.0));
      $amount = money($unit * $qty);

      $desc = trim((string)($li['description'] ?? 'Net32 Item'));
      $vpCode = trim((string)($li['vpCode'] ?? ''));
      $brand = trim((string)($li['brandName'] ?? ''));
      $mfg = trim((string)($li['manufacturerName'] ?? ''));

      $meta = [];
      if ($vpCode !== '') $meta[] = "VP: {$vpCode}";
      if ($brand !== '') $meta[] = "Brand: {$brand}";
      if ($mfg !== '') $meta[] = "Mfg: {$mfg}";

      //$fullDesc = $desc . ($meta ? " (" . implode(", ", $meta) . ")" : "");
      $fullDesc = $desc;

      $lines[] = [
        'sku' =>  $vpCode,
        'mpid' => (string)($li['mpid'] ?? ''),
        'vpCode' => $vpCode,
        'description' => $fullDesc,
        'qty' => $qty,
        'unit_price' => $unit,
        'amount' => $amount,
      ];
    }
  }

  // ✅ License: store in normalized for QBO custom field mapping later
  $license = net32_pick_first_license($o);

  return [
    'source' => 'NET32',
    'external_order_id' => $externalOrderId,
    'order_time' => (string)($o['orderTime'] ?? ''),
    'shipment_due_date' => (string)($o['shipmentDueDate'] ?? ''),
    'promised_delivery_date' => (string)($o['promisedDeliveryDate'] ?? ''),
    'customer' => [
      'name' => $customerName !== '' ? $customerName : $shipTo['name'],
      'email' => (string)($o['customerEmail'] ?? ''),
      'phone' => $shipTo['phone'],
    ],
    'ship_to' => $shipTo,
    'bill_to' => $billTo,

    // ✅ normalized license block (null or array)
    'license' => $license,

    'lines' => $lines,
    'totals' => [
      'shipping' => $shippingSubtotal,
      'tax' => $estimatedTax,
      'order_total' => $orderTotal,
    ],
    'raw_meta' => [
      'isFirstOrder' => $o['isFirstOrder'] ?? null,
    ],
  ];
}
