<?php
// lib/queue/processor.php

function queue_process_once(array $config, PDO $pdo): int {

  // Batch size and max attempts from config (with defaults)
  $batchSize   = (int)($config['queue']['batch_size'] ?? 25);
  $maxAttempts = (int)($config['queue']['max_attempts'] ?? 10);

  // Get Orders eligible for processing
  $orders = orders_fetch_eligible($pdo, $batchSize, $maxAttempts);
  log_line($config, "Queue eligible: " . count($orders));
  if (!count($orders)) return 0;

  $processed = 0;
  $dryRun = (!empty($config['dry_run']) || getenv('DRY_RUN') === '1');

  foreach ($orders as $row) {
    $orderId = (int)$row['id'];

    // Skipping if we fail to claim (another worker may have claimed it, or DB error)
    if (!orders_claim($pdo, $orderId, $maxAttempts)) {
      continue;
    }

    $sourceType = (string)$row['source_type'];
    $externalId = (string)$row['external_order_id'];

    log_line($config, "--- Processing order id={$orderId} source={$sourceType} externalOrderId={$externalId} ---");

    order_event($pdo, $orderId, 'PROCESSING', json_encode([
      'source_type' => $sourceType,
      'external_order_id' => $externalId,
      'attempt' => ((int)$row['attempt_count']) + 1,
    ], JSON_UNESCAPED_SLASHES));

    try {
      $raw = json_decode((string)$row['raw_payload_json'], true);
      if (!is_array($raw)) throw new Exception("raw_payload_json is not valid JSON");

      // Source normalize (registry-style)
      $normalized = source_normalize($sourceType, $raw);
      orders_store_normalized($pdo, $orderId, $normalized);

      order_event($pdo, $orderId, 'NORMALIZED', json_encode([
        'total' => $normalized['totals']['order_total'] ?? null,
        'lines' => is_array($normalized['lines'] ?? null) ? count($normalized['lines']) : 0,
      ], JSON_UNESCAPED_SLASHES));

      // Create/find customer (UPDATED: no token param)
      $customerId = qbo_find_or_create_customer($config, $pdo, $orderId, $normalized);

      // Build invoice payload (UPDATED: no token param)
      $invoicePayload = qbo_build_invoice_payload($config, $pdo, $orderId, $normalized, (string)$customerId);

      if ($dryRun) {
        log_line($config, "DRY RUN: would create invoice payload=" . json_encode($invoicePayload, JSON_UNESCAPED_SLASHES));

        orders_mark_processed($pdo, $orderId, 'INVOICE', null, (string)$customerId);

        order_event($pdo, $orderId, 'DRY_RUN_PROCESSED', json_encode([
          'qbo_customer_id' => (string)$customerId
        ], JSON_UNESCAPED_SLASHES));

        $processed++;
        continue;
      }

      $realm = (string)($config['qbo']['realm_id'] ?? '');
      if ($realm === '') throw new Exception("Missing QBO realm_id");

      $path = "/v3/company/{$realm}/invoice";

      // Create invoice (UPDATED: qbo_post() now fetches token internally)
      $resp = qbo_post($config, $pdo, $orderId, $path, $invoicePayload);

      $invoiceId = $resp['Invoice']['Id'] ?? null;
      if (!$invoiceId) throw new Exception("QBO created invoice but Invoice.Id missing");

      orders_mark_processed($pdo, $orderId, 'INVOICE', (string)$invoiceId, (string)$customerId);

      order_event($pdo, $orderId, 'QBO_INVOICE_CREATED', json_encode([
        'invoice_id' => (string)$invoiceId,
        'customer_id' => (string)$customerId,
      ], JSON_UNESCAPED_SLASHES));

      $processed++;

    } catch (Throwable $e) {
      $msg = $e->getMessage();
      log_line($config, "ERROR orderId={$orderId}: {$msg}");

      $attemptNow  = orders_get_attempt_count($pdo, $orderId);
      $nextRetryAt = compute_next_retry_at($config, $attemptNow);

      orders_mark_failed($pdo, $orderId, $msg, $nextRetryAt);

      order_event($pdo, $orderId, 'FAILED', json_encode([
        'message' => $msg,
        'next_retry_at' => $nextRetryAt,
        'attempt_count' => $attemptNow,
      ], JSON_UNESCAPED_SLASHES));
    }
  }

  return $processed;
}
