<?php
// lib/qbo/customer.php

// require_once __DIR__ . '/api.php';
// require_once __DIR__ . '/address.php';
// require_once __DIR__ . '/util.php';

function qbo_norm_email(string $email): string {
  $email = trim(strtolower($email));
  return $email;
}

function qbo_norm_phone(string $phone): string {
  // keep digits only for matching consistency
  return preg_replace('/\D+/', '', $phone);
}

function qbo_customer_display_name(array $n): string {
  $shipName = trim((string)($n['ship_to']['name'] ?? ''));
  $custName = trim((string)($n['customer']['name'] ?? ''));

  if ($shipName !== '' && $custName !== '' && $shipName !== $custName) return "{$shipName} - {$custName}";
  if ($shipName !== '') return $shipName;
  if ($custName !== '') return $custName;

  $ext = trim((string)($n['external_order_id'] ?? ''));
  return $ext !== '' ? "Customer {$ext}" : ("Customer " . uniqid());
}

/**
 * Try find a customer by email/phone/displayname, then create if missing.
 */
function qbo_find_or_create_customer(array $config, PDO $pdo, int $orderRowId, array $normalized): string {
  $realm = (string)($config['qbo']['realm_id'] ?? '');
  if ($realm === '') throw new Exception("Missing QBO realm_id");

  $emailRaw = (string)($normalized['customer']['email'] ?? '');
  $phoneRaw = (string)($normalized['customer']['phone'] ?? '');

  $email = $emailRaw !== '' ? qbo_norm_email($emailRaw) : '';
  $phoneDigits = $phoneRaw !== '' ? qbo_norm_phone($phoneRaw) : '';

  // 1) Match by email
  if ($email !== '') {
    $emailEsc = qbo_escape($email);
    // QBO supports querying nested fields like this in many cases
    $q = "select Id, DisplayName from Customer where PrimaryEmailAddr.Address = '{$emailEsc}' maxresults 1";
    $qr = qbo_query($config, $pdo, $orderRowId, $q);
    $id = qbo_first_id($qr, 'Customer');
    if ($id) {
      order_event($pdo, $orderRowId, 'QBO_CUSTOMER_FOUND_EMAIL', json_encode(['customer_id' => $id, 'email' => $email], JSON_UNESCAPED_SLASHES));
      return $id;
    }
  }

  // 2) Match by phone (digits)
  if ($phoneDigits !== '') {
    // QBO stores phone formatted; exact matching is tricky.
    // We attempt a basic match first; if it proves unreliable, we move matching to our own DB map later.
    $phoneEsc = qbo_escape($phoneRaw);
    $q = "select Id, DisplayName from Customer where PrimaryPhone.FreeFormNumber = '{$phoneEsc}' maxresults 1";
    $qr = qbo_query($config, $pdo, $orderRowId, $q);
    $id = qbo_first_id($qr, 'Customer');
    if ($id) {
      order_event($pdo, $orderRowId, 'QBO_CUSTOMER_FOUND_PHONE', json_encode(['customer_id' => $id, 'phone' => $phoneRaw], JSON_UNESCAPED_SLASHES));
      return $id;
    }
  }

  // 3) Fallback: DisplayName
  $displayName = qbo_customer_display_name($normalized);
  $dnEsc = qbo_escape($displayName);
  $q = "select Id, DisplayName from Customer where DisplayName = '{$dnEsc}' maxresults 1";
  $qr = qbo_query($config, $pdo, $orderRowId, $q);
  $id = qbo_first_id($qr, 'Customer');
  if ($id) {
    order_event($pdo, $orderRowId, 'QBO_CUSTOMER_FOUND_NAME', json_encode(['customer_id' => $id, 'display_name' => $displayName], JSON_UNESCAPED_SLASHES));
    return $id;
  }

  // Create new
  $payload = ['DisplayName' => $displayName];

  if ($phoneRaw !== '') $payload['PrimaryPhone'] = ['FreeFormNumber' => $phoneRaw];
  if ($email !== '') $payload['PrimaryEmailAddr'] = ['Address' => $email];

  $bill = is_array($normalized['bill_to'] ?? null) ? $normalized['bill_to'] : [];
  $addr = qbo_build_address($bill);
  if ($addr) $payload['BillAddr'] = $addr;

  $created = qbo_post($config, $pdo, $orderRowId, "/v3/company/{$realm}/customer", $payload);

  $createdId = $created['Customer']['Id'] ?? null;
  if (!$createdId) throw new Exception("Failed to extract created Customer.Id");

  order_event($pdo, $orderRowId, 'QBO_CUSTOMER_CREATED', json_encode([
    'customer_id' => (string)$createdId,
    'email' => $email ?: null,
    'phone' => $phoneRaw ?: null,
    'display_name' => $displayName,
  ], JSON_UNESCAPED_SLASHES));

  return (string)$createdId;
}
