<?php
// lib/qbo/auth.php

require_once __DIR__ . '/tokens.php';

/**
 * Get a valid access token.
 * - Reads tokens from DB
 * - Refreshes if expiring soon
 * - Writes updated tokens back to DB
 */
function qbo_get_access_token(array $config, PDO $pdo): string {
  $realmId = (string)($config['qbo']['realm_id'] ?? '');
  if ($realmId === '') throw new Exception("Missing QBO realm_id");

  $leeway = (int)($config['qbo']['token_refresh_leeway_sec'] ?? 180);

  $row = qbo_tokens_get($pdo, $realmId);
  if (!$row) {
    throw new Exception("No QBO tokens found in DB for realm_id={$realmId}. Seed qbo_tokens first.");
  }

  $accessToken  = (string)$row['access_token'];
  $refreshToken = (string)$row['refresh_token'];
  $expiresAt    = (string)$row['expires_at'];

  // If access token still good, use it
  if ($accessToken !== '' && $expiresAt !== '' && !qbo_tokens_is_expiring_soon($expiresAt, $leeway)) {
    return $accessToken;
  }

  // Otherwise refresh
  $data = qbo_refresh_access_token($config, $refreshToken);

  $newAccess  = (string)($data['access_token'] ?? '');
  $newRefresh = (string)($data['refresh_token'] ?? $refreshToken); // Intuit typically rotates refresh_token
  $expiresIn  = (int)($data['expires_in'] ?? 3600);

  if ($newAccess === '') throw new Exception("QBO refresh returned empty access_token");

  // Store new expiry
  $newExpiresAt = gmdate('Y-m-d H:i:s', time() + $expiresIn);

  qbo_tokens_upsert($pdo, $realmId, $newAccess, $newRefresh, $newExpiresAt);

  return $newAccess;
}

/**
 * Refresh QBO token using a provided refresh token (from DB).
 * Returns the raw Intuit token response array (must include access_token, usually refresh_token, expires_in).
 */
function qbo_refresh_access_token(array $config, string $refreshToken): array {
  $tokenUrl = "https://oauth.platform.intuit.com/oauth2/v1/tokens/bearer";

  $clientId = (string)($config['qbo']['client_id'] ?? '');
  $clientSecret = (string)($config['qbo']['client_secret'] ?? '');

  if ($clientId === '' || $clientSecret === '' || $refreshToken === '') {
    throw new Exception("Missing QBO client_id/client_secret/refresh_token");
  }

  $basic = base64_encode($clientId . ':' . $clientSecret);

  $headers = [
    "Authorization: Basic {$basic}",
    "Accept: application/json",
    "Content-Type: application/x-www-form-urlencoded",
  ];

  $body = http_build_query([
    'grant_type' => 'refresh_token',
    'refresh_token' => $refreshToken,
  ]);

  $resp = http_request('POST', $tokenUrl, $headers, $body);

  // Keep your existing logging
  log_http_file($config, 'qbo_refresh', redact_http($config, $resp));

  if ($resp['status'] < 200 || $resp['status'] >= 300) {
    throw new Exception("QBO token refresh failed HTTP {$resp['status']} body=" . truncate((string)$resp['body'], 1200));
  }

  $data = json_decode((string)$resp['body'], true);
  if (!is_array($data) || empty($data['access_token'])) {
    throw new Exception("QBO token refresh: missing access_token");
  }

  return $data;
}
