<?php
// lib/qbo/api.php

require_once __DIR__ . '/auth.php'; // provides qbo_get_access_token()

function qbo_query(array $config, PDO $pdo, int $orderRowId, string $query): array {
  $accessToken = qbo_get_access_token($config, $pdo);

  $base = rtrim((string)$config['qbo']['base_url'], '/');
  $realm = (string)$config['qbo']['realm_id'];
  $mv = qbo_minorversion($config);

  $url = "{$base}/v3/company/{$realm}/query?query=" . urlencode($query) . "&minorversion={$mv}";

  $headers = [
    "Authorization: Bearer {$accessToken}",
    "Accept: application/json",
  ];

  $resp = http_request('GET', $url, $headers, null);

  http_log_db($pdo, $orderRowId, 'qbo_query', redact_http($config, $resp));
  log_http_file($config, 'qbo_query', redact_http($config, $resp));

  if ($resp['status'] < 200 || $resp['status'] >= 300) {
    throw new Exception("QBO query failed HTTP {$resp['status']} body=" . truncate((string)$resp['body'], 1200));
  }

  $data = json_decode((string)$resp['body'], true);
  if (!is_array($data)) throw new Exception("QBO query returned non-JSON");
  return $data;
}

function qbo_post(array $config, PDO $pdo, int $orderRowId, string $path, array $payload): array {
  $accessToken = qbo_get_access_token($config, $pdo);

  $base = rtrim((string)$config['qbo']['base_url'], '/');
  $mv = qbo_minorversion($config);

  $sep = (strpos($path, '?') !== false) ? '&' : '?';
  $url = $base . $path . "{$sep}minorversion={$mv}";

  $headers = [
    "Authorization: Bearer {$accessToken}",
    "Accept: application/json",
    "Content-Type: application/json",
  ];

  $resp = http_request('POST', $url, $headers, json_encode($payload));

  http_log_db($pdo, $orderRowId, 'qbo_post', redact_http($config, $resp));
  log_http_file($config, 'qbo_post', redact_http($config, $resp));

  if ($resp['status'] < 200 || $resp['status'] >= 300) {
    throw new Exception("QBO POST failed HTTP {$resp['status']} body=" . truncate((string)$resp['body'], 2000));
  }

  $data = json_decode((string)$resp['body'], true);
  if (!is_array($data)) throw new Exception("QBO POST returned non-JSON");
  return $data;
}

function qbo_get(array $config, PDO $pdo, int $orderRowId, string $path): array {
  $accessToken = qbo_get_access_token($config, $pdo);

  $base = rtrim((string)$config['qbo']['base_url'], '/');
  $mv = qbo_minorversion($config);

  $sep = (strpos($path, '?') !== false) ? '&' : '?';
  $url = $base . $path . "{$sep}minorversion={$mv}";

  $headers = [
    "Authorization: Bearer {$accessToken}",
    "Accept: application/json",
  ];

  $resp = http_request('GET', $url, $headers, null);

  http_log_db($pdo, $orderRowId, 'qbo_get', redact_http($config, $resp));
  log_http_file($config, 'qbo_get', redact_http($config, $resp));

  if ($resp['status'] < 200 || $resp['status'] >= 300) {
    throw new Exception("QBO GET failed HTTP {$resp['status']} body=" . truncate((string)$resp['body'], 2000));
  }

  $data = json_decode((string)$resp['body'], true);
  if (!is_array($data)) throw new Exception("QBO GET returned non-JSON");
  return $data;
}

function qbo_first_id(array $qboResp, string $entityName): ?string {
  $qr = $qboResp['QueryResponse'] ?? null;
  if (!is_array($qr)) return null;
  $arr = $qr[$entityName] ?? null;
  if (!is_array($arr) || !count($arr)) return null;
  $first = $arr[0] ?? null;
  if (!is_array($first)) return null;
  return isset($first['Id']) ? (string)$first['Id'] : null;
}
