<?php
// admin/orders.php
// Modern ops UI for the queue.
// IMPORTANT: Protect this directory (Basic Auth / IP allowlist / move outside public_html)
// error_reporting(E_ALL);
// ini_set('display_errors', 1);


require $_SERVER['DOCUMENT_ROOT'] . '/net-32-qbo-sync/src/Support/env.php';
$config = require  $_SERVER['DOCUMENT_ROOT'] . '/net-32-qbo-sync/config.php';
require $_SERVER['DOCUMENT_ROOT'] . '/net-32-qbo-sync/db.php';
require __DIR__ . '/../layouts/app.php';

// ... your existing orders logic here ...
date_default_timezone_set($config['timezone'] ?? 'UTC');
$pdo = db($config);

function h($s): string { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function qs(array $extra = []): string {
  $q = $_GET;
  foreach ($extra as $k => $v) {
    if ($v === null) unset($q[$k]);
    else $q[$k] = $v;
  }
  return '?' . http_build_query($q);
}

function fmt_dt(?string $dt): string {
  if (!$dt) return '';
  try {
    $d = new DateTime($dt);
    return $d->format('M j, Y g:i A');
  } catch (Throwable $e) {
    return (string)$dt;
  }
}

function fmt_money($n): string {
  if ($n === null || $n === '') return '';
  if (!is_numeric($n)) return (string)$n;
  return '$' . number_format((float)$n, 2);
}

function short_err(?string $s, int $max = 140): string {
  if (!$s) return '';
  $s = trim($s);
  if (mb_strlen($s) <= $max) return $s;
  return mb_substr($s, 0, $max) . '…';
}

function json_decode_safe(?string $s): array {
  if (!$s) return [];
  $j = json_decode($s, true);
  return is_array($j) ? $j : [];
}

function json_get(array $arr, string $path) {
  $cur = $arr;
  foreach (explode('.', $path) as $k) {
    if (!is_array($cur) || !array_key_exists($k, $cur)) return null;
    $cur = $cur[$k];
  }
  return $cur;
}

function json_first(array $arr, array $paths) {
  foreach ($paths as $p) {
    $v = json_get($arr, $p);
    if ($v !== null && $v !== '' && !(is_array($v) && !count($v))) return $v;
  }
  return null;
}

function extract_order_display(array $order): array {
  $norm = json_decode_safe($order['normalized_json'] ?? '');
  $raw  = json_decode_safe($order['raw_payload_json'] ?? '');

  $name = (string)(json_first($norm, ['customer.name']) ?? '');
  $email = (string)(json_first($norm, ['customer.email']) ?? '');
  $phone = (string)(json_first($norm, ['customer.phone']) ?? '');

  $company = (string)(json_first($norm, ['ship_to.name']) ?? '');

  $total = json_first($norm, ['totals.order_total']);
  $placedAt = (string)(json_first($norm, ['order_time']) ?? '');

  if (!$name)    $name = (string)($raw['customerName'] ?? '');
  if (!$phone)   $phone = (string)($raw['shippingAddress']['phone'] ?? '');
  if (!$company) $company = (string)($raw['shippingAddress']['name'] ?? '');
  if ($total === null || $total === '') $total = $raw['orderTotal'] ?? null;
  if (!$placedAt) $placedAt = (string)($raw['orderTime'] ?? '');

  if (!$placedAt) $placedAt = (string)($order['created_at'] ?? '');
  if (!$placedAt) $placedAt = (string)($order['updated_at'] ?? '');

  $username = (string)(json_first($norm, ['customer.username','user.username','account.username']) ?? '');

  $shipCity = (string)(json_first($norm, ['ship_to.city']) ?? '');
  $shipRegion = (string)(json_first($norm, ['ship_to.region']) ?? '');
  $shipPostal = (string)(json_first($norm, ['ship_to.postalCode']) ?? '');
  $shipLine = trim(implode(', ', array_filter([$shipCity, $shipRegion, $shipPostal])));

  $isFirstOrder = null;
  if (array_key_exists('raw_meta', $norm) && is_array($norm['raw_meta']) && array_key_exists('isFirstOrder', $norm['raw_meta'])) {
    $isFirstOrder = (bool)$norm['raw_meta']['isFirstOrder'];
  } elseif (array_key_exists('isFirstOrder', $raw)) {
    $isFirstOrder = (bool)$raw['isFirstOrder'];
  }

  return [
    'name' => $name,
    'email' => $email,
    'phone' => $phone,
    'username' => $username,
    'company' => $company,
    'total' => $total,
    'placed_at' => $placedAt,
    'ship_line' => $shipLine,
    'is_first_order' => $isFirstOrder,
  ];
}

function badge_class(string $st): string {
  return match ($st) {
    'RECEIVED' => 'badge blue',
    'PROCESSING' => 'badge purple',
    'PROCESSED' => 'badge green',
    'FAILED' => 'badge red',
    'IGNORED' => 'badge gray',
    default => 'badge gray',
  };
}

// actions
$action = $_GET['action'] ?? '';
$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if ($action === 'retry' && $id > 0) {
  queue_retry_order($config, $pdo, $id, false);
  header("Location: orders.php?msg=" . urlencode("Order #{$id} queued for retry"));
  exit;
}

$msg = $_GET['msg'] ?? '';

$status = $_GET['status'] ?? '';
$sourceId = $_GET['source_id'] ?? '';
$q = trim((string)($_GET['q'] ?? ''));

$sources = db_all($pdo, "SELECT id, type, name FROM sources ORDER BY id ASC");

// Filters
$where = [];
$params = [];

if ($status !== '') {
  $where[] = "o.status = :status";
  $params[':status'] = $status;
}
if ($sourceId !== '' && ctype_digit((string)$sourceId)) {
  $where[] = "o.source_id = :source_id";
  $params[':source_id'] = (int)$sourceId;
}
if ($q !== '') {
  $where[] = "(o.external_order_id LIKE :q OR o.qbo_txn_id LIKE :q)";
  $params[':q'] = '%' . $q . '%';
}
$whereSql = count($where) ? ("WHERE " . implode(" AND ", $where)) : "";

// Stats (respect filter except status)
$statsWhere = [];
$statsParams = [];
if ($sourceId !== '' && ctype_digit((string)$sourceId)) {
  $statsWhere[] = "o.source_id = :source_id";
  $statsParams[':source_id'] = (int)$sourceId;
}
if ($q !== '') {
  $statsWhere[] = "(o.external_order_id LIKE :q OR o.qbo_txn_id LIKE :q)";
  $statsParams[':q'] = '%' . $q . '%';
}
$statsSql = count($statsWhere) ? ("WHERE " . implode(" AND ", $statsWhere)) : "";

$statsRows = db_all($pdo, "
  SELECT o.status, COUNT(*) AS c
  FROM orders o
  {$statsSql}
  GROUP BY o.status
", $statsParams);

$stats = [
  'RECEIVED' => 0,
  'PROCESSING' => 0,
  'PROCESSED' => 0,
  'FAILED' => 0,
  'IGNORED' => 0,
];
foreach ($statsRows as $sr) {
  $st = (string)$sr['status'];
  if (isset($stats[$st])) $stats[$st] = (int)$sr['c'];
}
$totalAll = array_sum($stats);

// Orders list (max 200)
$rows = db_all($pdo, "
  SELECT o.*, s.type AS source_type, s.name AS source_name
  FROM orders o
  JOIN sources s ON s.id = o.source_id
  {$whereSql}
  ORDER BY o.updated_at DESC
  LIMIT 200
", $params);


render_admin_page($config, $pdo, 'Orders', function() use (
  $config, $pdo,
  $rows, $sources,
  $status, $sourceId, $q,
  $stats, $totalAll,
  $msg
) {
  ?>
  <div class="topbar">
    <div class="wrap">
      <div class="title-row">
        <div>
          <h1>Integrations • Orders Queue</h1>
          <div class="sub">Showing up to 200 newest records. Use filters to narrow down.</div>
        </div>
        <div class="pill">
          <span class="dot" style="background: var(--accent)"></span>
          <span class="mono">DB: <?= h($config['db']['name'] ?? '') ?></span>
        </div>
      </div>

      <div class="grid">
        <div class="stat"><div class="k">Total (filtered)</div><div class="v"><?= (int)$totalAll ?></div></div>
        <div class="stat"><div class="k">Received</div><div class="v"><?= (int)$stats['RECEIVED'] ?></div></div>
        <div class="stat"><div class="k">Processing</div><div class="v"><?= (int)$stats['PROCESSING'] ?></div></div>
        <div class="stat"><div class="k">Processed</div><div class="v"><?= (int)$stats['PROCESSED'] ?></div></div>
        <div class="stat"><div class="k">Failed</div><div class="v"><?= (int)$stats['FAILED'] ?></div></div>
      </div>

      <div class="filters">
        <form method="get" action="orders.php">
          <div>
            <label>Source</label>
            <select name="source_id">
              <option value="">All sources</option>
              <?php foreach ($sources as $s): ?>
                <option value="<?= (int)$s['id'] ?>" <?= ((string)$sourceId === (string)$s['id']) ? 'selected' : '' ?>>
                  <?= h($s['type']) ?> — <?= h($s['name']) ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>

          <div>
            <label>Status</label>
            <select name="status">
              <option value="">All statuses</option>
              <?php foreach (['RECEIVED','PROCESSING','PROCESSED','FAILED','IGNORED'] as $st): ?>
                <option value="<?= h($st) ?>" <?= ($status === $st) ? 'selected' : '' ?>><?= h($st) ?></option>
              <?php endforeach; ?>
            </select>
          </div>

          <div>
            <label>Search</label>
            <input type="text" name="q" value="<?= h($q) ?>" placeholder="external_order_id or qbo_txn_id">
          </div>

          <div class="right">
            <button class="btn" type="submit">Apply</button>
            <a class="btn btn-ghost" style="text-decoration:none;display:inline-flex;align-items:center;justify-content:center;margin-left:8px"
               href="orders.php">Reset</a>
          </div>
        </form>
      </div>

      <?php if ($msg): ?>
        <div class="msg"><?= h($msg) ?></div>
      <?php endif; ?>
    </div>
  </div>

  <div class="wrap">
    <div class="panel">
      <div class="panel-h">
        <div>
          <div style="font-weight:800">Latest Orders</div>
          <div class="small">Tip: click “View” to open tabs (Overview/Raw/Normalized/Events/HTTP).</div>
        </div>
        <div class="small muted">
          Quick links:
          <a href="<?= h(qs(['status'=>'FAILED'])) ?>">Failed</a> •
          <a href="<?= h(qs(['status'=>'RECEIVED'])) ?>">Received</a> •
          <a href="<?= h(qs(['status'=>'PROCESSED'])) ?>">Processed</a>
        </div>
      </div>

      <div class="tablewrap">
        <table class="table">
          <thead>
            <tr>
              <th style="width:70px">ID</th>
              <th>External Order</th>
              <th>Customer</th>
              <th style="width:120px">Total</th>
              <th>Status</th>
              <th style="width:120px">Attempts</th>
              <th style="width:140px">QBO</th>
              <th>Last Error</th>
              <th style="width:170px">Placed</th>
              <th style="width:170px">Updated</th>
              <th>Source</th>
              <th style="width:160px">Actions</th>
            </tr>
          </thead>
          <tbody>
 
            <?php foreach ($rows as $r): ?>
              <?php
                $st = (string)$r['status'];
                $d = extract_order_display($r);
              ?>
              <tr>
                <td class="mono"><?= (int)$r['id'] ?></td>
                <td class="mono"><?= h($r['external_order_id']) ?></td>

                <td>
                  <div style="font-weight:900"><?= h($d['name'] ?: '(no name)') ?></div>
                  <div class="muted" style="font-size:12px">
                    <?= h($d['company'] ?: '') ?>
                    <?php if ($d['company'] && ($d['email'] || $d['phone'])): ?> · <?php endif; ?>
                    <?= h($d['email'] ?: '') ?>
                    <?php if ($d['email'] && $d['phone']): ?> · <?php endif; ?>
                    <?= h($d['phone'] ?: '') ?>
                    <?php if (!$d['company'] && !$d['email'] && !$d['phone']): ?>(no contact info)<?php endif; ?>
                  </div>
                  <?php if ($d['ship_line']): ?>
                    <div class="muted" style="font-size:12px">Ship: <?= h($d['ship_line']) ?></div>
                  <?php endif; ?>
                </td>

                <td class="mono" style="font-weight:900"><?= h(fmt_money($d['total'])) ?></td>

                <td>
                  <span class="<?= h(badge_class($st)) ?>">
                    <span class="dot"></span><?= h($st) ?>
                  </span>
                </td>

                <td>
                  <div class="mono"><?= (int)$r['attempt_count'] ?></div>
                  <?php if (!empty($r['next_retry_at'])): ?>
                    <div class="muted" style="font-size:12px">next: <?= h(fmt_dt($r['next_retry_at'])) ?></div>
                  <?php endif; ?>
                </td>

                <td>
                  <?php if (!empty($r['qbo_txn_id'])): ?>
                    <div style="font-weight:800"><?= h($r['qbo_txn_type']) ?></div>
                    <div class="mono"><?= h($r['qbo_txn_id']) ?></div>
                  <?php else: ?>
                    <span class="muted">(none)</span>
                  <?php endif; ?>
                </td>

                <td style="max-width:420px">
                  <?php if (!empty($r['error_message'])): ?>
                    <span class="muted"><?= h(short_err((string)$r['error_message'])) ?></span>
                  <?php else: ?>
                    <span class="muted">(none)</span>
                  <?php endif; ?>
                </td>

                <td class="muted" style="font-size:12px"><?= h(fmt_dt($d['placed_at'])) ?></td>
                <td class="muted" style="font-size:12px"><?= h(fmt_dt($r['updated_at'])) ?></td>

                <td>
                  <div style="font-weight:800"><?= h($r['source_type']) ?></div>
                  <div class="muted" style="font-size:12px"><?= h($r['source_name']) ?></div>
                </td>

                <td>
                  <div class="actions">
                    <a class="btn btn-ghost" style="text-decoration:none"
                       href="order_view.php?id=<?= (int)$r['id'] ?>&tab=overview">View</a>

                    <?php if ($st === 'FAILED' || $st === 'RECEIVED'): ?>
                      <a class="btn btn-danger" style="text-decoration:none"
                         href="orders.php?action=retry&id=<?= (int)$r['id'] ?>"
                         onclick="return confirm('Retry this order now?');">Retry</a>
                    <?php endif; ?>
                  </div>
                </td>
              </tr>
            <?php endforeach; ?>

            <?php if (!count($rows)): ?>
              <tr><td colspan="12" class="muted">No results with current filters.</td></tr>
            <?php endif; ?>
          </tbody>
        </table>
      </div>

      <div class="cards">
        <?php foreach ($rows as $r): ?>
          <?php
            $st = (string)$r['status'];
            $d = extract_order_display($r);
          ?>
          <div class="card">
            <div class="row">
              <div>
                <div class="muted k">Order</div>
                <div class="v mono"><?= h($r['external_order_id']) ?></div>
              </div>
              <div style="text-align:right">
                <div class="muted k">Row ID</div>
                <div class="v mono"><?= (int)$r['id'] ?></div>
              </div>
            </div>

            <div class="row">
              <div>
                <div class="muted k">Customer</div>
                <div class="v"><b><?= h($d['name'] ?: '(no name)') ?></b></div>
                <div class="muted" style="font-size:12px">
                  <?= h($d['company'] ?: '') ?>
                  <?php if ($d['company'] && ($d['email'] || $d['phone'])): ?> · <?php endif; ?>
                  <?= h($d['email'] ?: '') ?>
                  <?php if ($d['email'] && $d['phone']): ?> · <?php endif; ?>
                  <?= h($d['phone'] ?: '') ?>
                </div>
              </div>
            </div>

            <div class="row">
              <div>
                <div class="muted k">Total</div>
                <div class="v mono" style="font-weight:900"><?= h(fmt_money($d['total'])) ?></div>
              </div>
              <div style="text-align:right">
                <div class="muted k">Placed</div>
                <div class="v muted"><?= h(fmt_dt($d['placed_at'])) ?></div>
              </div>
            </div>

            <div class="row">
              <div>
                <span class="<?= h(badge_class($st)) ?>"><span class="dot"></span><?= h($st) ?></span>
              </div>
              <div style="text-align:right">
                <div class="muted k">Attempts</div>
                <div class="v mono"><?= (int)$r['attempt_count'] ?></div>
              </div>
            </div>

            <?php if (!empty($r['error_message'])): ?>
              <div class="row">
                <div>
                  <div class="muted k">Last Error</div>
                  <div class="v muted"><?= h(short_err((string)$r['error_message'], 220)) ?></div>
                </div>
              </div>
            <?php endif; ?>

            <div class="row">
              <div class="muted k">Updated</div>
              <div class="v muted"><?= h(fmt_dt($r['updated_at'])) ?></div>
            </div>

            <div class="row">
              <div>
                <div class="muted k">Source</div>
                <div class="v"><b><?= h($r['source_type']) ?></b> — <span class="muted"><?= h($r['source_name']) ?></span></div>
              </div>
            </div>

            <div class="actions">
              <a class="btn btn-ghost" style="text-decoration:none"
                 href="order_view.php?id=<?= (int)$r['id'] ?>&tab=overview">View</a>
              <?php if ($st === 'FAILED' || $st === 'RECEIVED'): ?>
                <a class="btn btn-danger" style="text-decoration:none"
                   href="orders.php?action=retry&id=<?= (int)$r['id'] ?>"
                   onclick="return confirm('Retry this order now?');">Retry</a>
              <?php endif; ?>
            </div>
          </div>
        <?php endforeach; ?>

        <?php if (!count($rows)): ?>
          <div class="card"><div class="muted">No results with current filters.</div></div>
        <?php endif; ?>
      </div>
    </div>

    <div style="height:40px"></div>
  </div>
<?php
}, [
  'active' => 'orders',
  'basePath' => rtrim(dirname($_SERVER['SCRIPT_NAME']), '/'),
]);