<?php
// admin/order_view.php
// Single order details view (tabs) — JSON pretty-print + improved <pre> formatting
// IMPORTANT: Protect this directory (Basic Auth / IP allowlist / move outside public_html)
require $_SERVER['DOCUMENT_ROOT'] . '/net-32-qbo-sync/src/Support/env.php';
$config = require  $_SERVER['DOCUMENT_ROOT'] . '/net-32-qbo-sync/config.php';
require $_SERVER['DOCUMENT_ROOT'] . '/net-32-qbo-sync/db.php';
require __DIR__ . '/../layouts/app.php';

date_default_timezone_set($config['timezone'] ?? 'UTC');
$pdo = db($config);

function h($s): string { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

function fmt_dt(?string $dt): string {
  if (!$dt) return '';
  try {
    $d = new DateTime($dt);
    return $d->format('M j, Y g:i A');
  } catch (Throwable $e) {
    return (string)$dt;
  }
}

function fmt_money($n): string {
  if ($n === null || $n === '') return '';
  if (!is_numeric($n)) return (string)$n;
  return '$' . number_format((float)$n, 2);
}

function json_decode_safe(?string $s): array {
  if (!$s) return [];
  $j = json_decode($s, true);
  return is_array($j) ? $j : [];
}

function json_get(array $arr, string $path) {
  $cur = $arr;
  foreach (explode('.', $path) as $k) {
    if (!is_array($cur) || !array_key_exists($k, $cur)) return null;
    $cur = $cur[$k];
  }
  return $cur;
}

function json_first(array $arr, array $paths) {
  foreach ($paths as $p) {
    $v = json_get($arr, $p);
    if ($v !== null && $v !== '' && !(is_array($v) && !count($v))) return $v;
  }
  return null;
}

function extract_order_display(array $order): array {
  $norm = json_decode_safe($order['normalized_json'] ?? '');
  $raw  = json_decode_safe($order['raw_payload_json'] ?? '');

  $name = (string)(json_first($norm, ['customer.name']) ?? '');
  $email = (string)(json_first($norm, ['customer.email']) ?? '');
  $phone = (string)(json_first($norm, ['customer.phone']) ?? '');
  $company = (string)(json_first($norm, ['ship_to.name']) ?? '');

  $total = json_first($norm, ['totals.order_total']);
  $placedAt = (string)(json_first($norm, ['order_time']) ?? '');

  if (!$name)    $name = (string)($raw['customerName'] ?? '');
  if (!$phone)   $phone = (string)($raw['shippingAddress']['phone'] ?? '');
  if (!$company) $company = (string)($raw['shippingAddress']['name'] ?? '');
  if ($total === null || $total === '') $total = $raw['orderTotal'] ?? null;
  if (!$placedAt) $placedAt = (string)($raw['orderTime'] ?? '');

  if (!$placedAt) $placedAt = (string)($order['created_at'] ?? '');
  if (!$placedAt) $placedAt = (string)($order['updated_at'] ?? '');

  $username = (string)(json_first($norm, ['customer.username','user.username','account.username']) ?? '');

  $shipCity = (string)(json_first($norm, ['ship_to.city']) ?? '');
  $shipRegion = (string)(json_first($norm, ['ship_to.region']) ?? '');
  $shipPostal = (string)(json_first($norm, ['ship_to.postalCode']) ?? '');
  $shipLine = trim(implode(', ', array_filter([$shipCity, $shipRegion, $shipPostal])));

  $isFirstOrder = null;
  if (array_key_exists('raw_meta', $norm) && is_array($norm['raw_meta']) && array_key_exists('isFirstOrder', $norm['raw_meta'])) {
    $isFirstOrder = (bool)$norm['raw_meta']['isFirstOrder'];
  } elseif (array_key_exists('isFirstOrder', $raw)) {
    $isFirstOrder = (bool)$raw['isFirstOrder'];
  }

  return [
    'name' => $name,
    'email' => $email,
    'phone' => $phone,
    'username' => $username,
    'company' => $company,
    'total' => $total,
    'placed_at' => $placedAt,
    'ship_line' => $shipLine,
    'is_first_order' => $isFirstOrder,
  ];
}

function badge_class(string $st): string {
  return match ($st) {
    'RECEIVED' => 'badge blue',
    'PROCESSING' => 'badge purple',
    'PROCESSED' => 'badge green',
    'FAILED' => 'badge red',
    'IGNORED' => 'badge gray',
    default => 'badge gray',
  };
}

/**
 * Pretty-print JSON if valid; otherwise return raw string.
 * - Keeps output safe (caller should escape).
 * - Handles already-pretty JSON fine.
 * - Optional maxBytes safety limit (0 = unlimited).
 */
function pretty_json(?string $json, int $maxBytes = 0): string {
  if ($json === null || $json === '') return '';

  if ($maxBytes > 0 && strlen($json) > $maxBytes) {
    // NOTE: truncation may break JSON validity — we intentionally show raw truncated data.
    return substr($json, 0, $maxBytes) . "\n/* truncated */";
  }

  $decoded = json_decode($json, true);
  if (json_last_error() !== JSON_ERROR_NONE) return $json;

  $pretty = json_encode(
    $decoded,
    JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE
  );

  return $pretty !== false ? $pretty : $json;
}

function pre_json(?string $json, string $fallbackLabel = '(none)', int $maxBytes = 0): string {
  $json = trim((string)$json);
  if ($json === '') return h($fallbackLabel);
  return h(pretty_json($json, $maxBytes));
}

// Inputs
$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$tab = $_GET['tab'] ?? 'overview';
$tabAllow = ['overview','raw','normalized','events','http'];
if (!in_array($tab, $tabAllow, true)) $tab = 'overview';

if ($id <= 0) {
  header("Location: orders.php?msg=" . urlencode("Missing order id"));
  exit;
}

// Load order
$order = db_one($pdo, "
  SELECT o.*, s.type AS source_type, s.name AS source_name
  FROM orders o
  JOIN sources s ON s.id = o.source_id
  WHERE o.id=?
", [$id]);

if (!$order) {
  header("Location: orders.php?msg=" . urlencode("Order not found"));
  exit;
}

// Retry (from this page)
$action = $_GET['action'] ?? '';
if ($action === 'retry' && $id > 0) {
  queue_retry_order($config, $pdo, $id, false);
  header("Location: order_view.php?id={$id}&tab=" . urlencode($tab) . "&msg=" . urlencode("Order #{$id} queued for retry"));
  exit;
}

$events = db_all($pdo, "SELECT * FROM order_events WHERE order_id=? ORDER BY created_at DESC LIMIT 200", [$id]);
$http   = db_all($pdo, "SELECT * FROM http_logs WHERE order_id=? ORDER BY created_at DESC LIMIT 80", [$id]);

$st = (string)$order['status'];
$badge = badge_class($st);
$d = extract_order_display($order);

$msg = $_GET['msg'] ?? '';

render_admin_page($config, $pdo, 'Order Details', function() use (
  $config, $pdo,
  $id, $tab,
  $order, $events, $http,
  $st, $badge, $d,
  $msg
) {
  ?>
  <div class="topbar">
    <div class="wrap">
      <div class="title-row">
        <div>
          <h1>Integrations • Order Details</h1>
          <div class="sub">Order #<?= (int)$id ?> • Tabs: Overview / Raw / Normalized / Events / HTTP</div>
        </div>
        <div class="pill">
          <span class="dot" style="background: var(--accent)"></span>
          <span class="mono">DB: <?= h($config['db']['name'] ?? '') ?></span>
        </div>
      </div>

      <?php if ($msg): ?>
        <div class="msg"><?= h($msg) ?></div>
      <?php endif; ?>
    </div>
  </div>

  <div class="wrap">
    <div class="panel">
      <div class="panel-h">
        <div>
          <div style="display:flex;align-items:center;gap:10px;flex-wrap:wrap">
            <div style="font-weight:900">Order #<?= (int)$id ?></div>
            <span class="<?= h($badge) ?>"><span class="dot"></span><?= h($st) ?></span>
            <span class="pill mono">External: <?= h($order['external_order_id']) ?></span>

            <?php if ($d['company']): ?>
              <span class="pill">🏢 <?= h($d['company']) ?></span>
            <?php endif; ?>
            <?php if ($d['username']): ?>
              <span class="pill mono">@<?= h($d['username']) ?></span>
            <?php endif; ?>
            <?php if ($d['is_first_order'] === true): ?>
              <span class="pill" style="border-color:rgba(34,197,94,.35);background:rgba(34,197,94,.10)">✨ First Order</span>
            <?php elseif ($d['is_first_order'] === false): ?>
              <span class="pill" style="border-color:rgba(148,163,184,.22);background:rgba(148,163,184,.08)">Returning</span>
            <?php endif; ?>
          </div>

          <div class="small">
            Customer: <b><?= h($d['name'] ?: '(no name)') ?></b>
            <?php if ($d['email']): ?> • <?= h($d['email']) ?><?php endif; ?>
            <?php if ($d['phone']): ?> • <?= h($d['phone']) ?><?php endif; ?>
            • Total: <span class="mono"><?= h(fmt_money($d['total'])) ?></span>
            • Placed: <span class="mono"><?= h(fmt_dt($d['placed_at'])) ?></span>
          </div>

          <div class="small muted">
            Source: <b><?= h($order['source_type']) ?></b> — <?= h($order['source_name']) ?>
            <?php if (!empty($order['qbo_txn_id'])): ?>
              • QBO: <b><?= h($order['qbo_txn_type']) ?></b> <span class="mono"><?= h($order['qbo_txn_id']) ?></span>
            <?php endif; ?>
          </div>
        </div>

        <div class="actions">
          <a class="btn btn-ghost" style="text-decoration:none" href="orders.php">Back</a>
          <?php if ($st === 'FAILED' || $st === 'RECEIVED'): ?>
            <a class="btn btn-danger" style="text-decoration:none"
               href="order_view.php?action=retry&id=<?= (int)$id ?>&tab=<?= h($tab) ?>"
               onclick="return confirm('Retry this order now?');">Retry Now</a>
          <?php endif; ?>
        </div>
      </div>

      <div class="tabs">
        <a class="tab <?= $tab==='overview'?'active':'' ?>" href="order_view.php?id=<?= (int)$id ?>&tab=overview">Overview</a>
        <a class="tab <?= $tab==='raw'?'active':'' ?>" href="order_view.php?id=<?= (int)$id ?>&tab=raw">Raw</a>
        <a class="tab <?= $tab==='normalized'?'active':'' ?>" href="order_view.php?id=<?= (int)$id ?>&tab=normalized">Normalized</a>
        <a class="tab <?= $tab==='events'?'active':'' ?>" href="order_view.php?id=<?= (int)$id ?>&tab=events">Events</a>
        <a class="tab <?= $tab==='http'?'active':'' ?>" href="order_view.php?id=<?= (int)$id ?>&tab=http">HTTP</a>
      </div>

      <div class="content">
        <?php if ($tab === 'overview'): ?>
          <div class="grid" style="grid-template-columns: repeat(3, minmax(0,1fr)); margin-top:0">
            <div class="stat">
              <div class="k">Customer</div>
              <div class="v" style="font-size:14px;font-weight:900"><?= h($d['name'] ?: '(no name)') ?></div>
              <div class="sub">
                <?php if ($d['company']): ?>🏢 <span class="mono"><?= h($d['company']) ?></span><br><?php endif; ?>
                <?= $d['email'] ? "Email: <span class='mono'>".h($d['email'])."</span><br>" : "" ?>
                <?= $d['phone'] ? "Phone: <span class='mono'>".h($d['phone'])."</span><br>" : "" ?>
                Placed: <span class="mono"><?= h(fmt_dt($d['placed_at'])) ?></span>
              </div>
            </div>

            <div class="stat">
              <div class="k">Order Total</div>
              <div class="v mono"><?= h(fmt_money($d['total'])) ?></div>
              <div class="sub">External: <span class="mono"><?= h($order['external_order_id']) ?></span></div>
              <?php if ($d['is_first_order'] === true): ?>
                <div class="sub">✨ First order</div>
              <?php elseif ($d['is_first_order'] === false): ?>
                <div class="sub">Returning customer</div>
              <?php endif; ?>
            </div>

            <div class="stat">
              <div class="k">Processing</div>
              <div class="v mono" style="font-size:14px;font-weight:800">
                Attempts: <?= (int)$order['attempt_count'] ?>
              </div>
              <div class="sub">Last attempt: <span class="mono"><?= h($order['last_attempt_at'] ? fmt_dt($order['last_attempt_at']) : '(none)') ?></span></div>
              <div class="sub">Updated: <span class="mono"><?= h($order['updated_at'] ? fmt_dt($order['updated_at']) : '(none)') ?></span></div>
              <?php if (!empty($order['next_retry_at'])): ?>
                <div class="sub">Next retry: <span class="mono"><?= h(fmt_dt($order['next_retry_at'])) ?></span></div>
              <?php endif; ?>
            </div>
          </div>

          <details open>
            <summary>Last error</summary>
            <pre><?= h($order['error_message'] ?: '(none)') ?></pre>
          </details>

          <details>
            <summary>Normalized quick peek</summary>
            <pre class="json pretty"><?= pre_json($order['normalized_json'], '(none)') ?></pre>
          </details>

        <?php elseif ($tab === 'raw'): ?>
          <details open>
            <summary>Raw payload JSON</summary>
            <pre class="json pretty"><?= pre_json($order['raw_payload_json'], '(none)') ?></pre>
          </details>

        <?php elseif ($tab === 'normalized'): ?>
          <details open>
            <summary>Normalized JSON</summary>
            <pre class="json pretty"><?= pre_json($order['normalized_json'], '(none)') ?></pre>
          </details>

        <?php elseif ($tab === 'events'): ?>
          <details open>
            <summary>Events (latest first)</summary>
            <table class="table" style="margin-top:10px">
              <thead><tr><th style="width:210px">Time</th><th style="width:220px">Type</th><th>Details</th></tr></thead>
              <tbody>
              <?php foreach ($events as $ev): ?>
                <tr>
                  <td class="muted mono"><?= h(fmt_dt($ev['created_at'])) ?></td>
                  <td><b><?= h($ev['event_type']) ?></b></td>
                  <td><pre class="json pretty"><?= pre_json($ev['details_json'] ?? '', '') ?></pre></td>
                </tr>
              <?php endforeach; ?>
              <?php if (!count($events)): ?>
                <tr><td colspan="3" class="muted">No events.</td></tr>
              <?php endif; ?>
              </tbody>
            </table>
          </details>

        <?php elseif ($tab === 'http'): ?>
          <details open>
            <summary>HTTP logs (latest 80)</summary>
            <table class="table" style="margin-top:10px">
              <thead>
                <tr>
                  <th style="width:190px">Time</th>
                  <th style="width:120px">Tag</th>
                  <th style="width:80px">Status</th>
                  <th>Request</th>
                  <th>Response</th>
                </tr>
              </thead>
              <tbody>
              <?php foreach ($http as $hl): ?>
                <tr>
                  <td class="muted mono"><?= h(fmt_dt($hl['created_at'])) ?></td>
                  <td><b><?= h($hl['tag']) ?></b></td>
                  <td class="mono"><?= h($hl['response_status']) ?></td>

                  <td style="max-width:520px">
                    <div class="muted mono" style="font-size:12px;margin-bottom:6px">
                      <?= h($hl['request_method']) ?> <?= h($hl['request_url']) ?>
                    </div>
                    <details>
                      <summary class="muted">Headers</summary>
                      <pre class="json pretty"><?= pre_json($hl['request_headers'] ?? '', '') ?></pre>
                    </details>
                    <details>
                      <summary class="muted">Body</summary>
                      <pre class="json pretty"><?= pre_json($hl['request_body'] ?? '', '') ?></pre>
                    </details>
                  </td>

                  <td style="max-width:520px">
                    <details>
                      <summary class="muted">Headers</summary>
                      <pre class="json pretty"><?= pre_json($hl['response_headers'] ?? '', '') ?></pre>
                    </details>
                    <details open>
                      <summary class="muted">Body</summary>
                      <pre class="json pretty"><?= pre_json($hl['response_body'] ?? '', '') ?></pre>
                    </details>
                  </td>
                </tr>
              <?php endforeach; ?>
              <?php if (!count($http)): ?>
                <tr><td colspan="5" class="muted">No HTTP logs.</td></tr>
              <?php endif; ?>
              </tbody>
            </table>
          </details>
        <?php endif; ?>

        <?php if (!empty($config['admin']['allow_force_rerun'])): ?>
          <form method="POST" action="/net-32-qbo-sync/admin/actions/order_force_rerun.php" style="display:inline;">
            <input type="hidden" name="id" value="<?= (int)$order['id'] ?>">
            <input type="hidden" name="csrf" value="<?= htmlspecialchars($_SESSION['csrf'] ?? '') ?>">
            <button type="submit" class="btn btn-warning"
              onclick="return confirm('Force re-run this order? This will queue it again and create another QBO invoice in Sandbox.');">
              Force Re-run (Sandbox)
            </button>
          </form>
        <?php endif; ?>
      </div>
    </div>

    <div style="height:40px"></div>
  </div>
<?php
}, [
  'active' => 'orders',
  'basePath' => rtrim(dirname($_SERVER['SCRIPT_NAME']), '/'),
]);