<?php
// admin/clients.php
// Lists all integration sources (NET32, WooCommerce, Amazon, etc.)
require $_SERVER['DOCUMENT_ROOT'] . '/net-32-qbo-sync/src/Support/env.php';
$config = require $_SERVER['DOCUMENT_ROOT'] . '/net-32-qbo-sync/config.php';
require $_SERVER['DOCUMENT_ROOT'] . '/net-32-qbo-sync/db.php';
require __DIR__ . '/../layouts/app.php';

date_default_timezone_set($config['timezone'] ?? 'UTC');
$pdo = db($config);

function h($s): string { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

function fmt_dt(?string $dt): string {
  if (!$dt) return '';
  try { return (new DateTime($dt))->format('M j, Y g:i A'); }
  catch (Throwable $e) { return (string)$dt; }
}

function json_decode_safe(?string $s): array {
  if (!$s) return [];
  $j = json_decode($s, true);
  return is_array($j) ? $j : [];
}

function badge_enabled(int $isEnabled): string {
  return $isEnabled ? 'badge green' : 'badge gray';
}

// Load sources
$sources = db_all($pdo, "
  SELECT
    id,
    type,
    name,
    is_enabled,
    poll_interval_seconds,
    last_polled_at,
    config_json,
    created_at,
    updated_at
  FROM sources
  ORDER BY id ASC
");

render_admin_page($config, $pdo, 'Clients', function() use ($config, $sources) {
  ?>
  <div class="topbar">
    <div class="wrap">
      <div class="title-row">
        <div>
          <h1>Integrations • Clients</h1>
          <div class="sub">All connected sources and integrations</div>
        </div>
        <div class="pill">
          <span class="dot" style="background: var(--accent)"></span>
          <span class="mono">DB: <?= h($config['db']['name'] ?? '') ?></span>
        </div>
      </div>
    </div>
  </div>

  <div class="filters">
    <a class="btn" style="text-decoration:none; display: inline-block;" href="client_new.php">+ New Client</a>
  </div>

  <div class="wrap">
    <div class="panel">
      <div class="panel-h">
        <div style="font-weight:800">Sources</div>
        <div class="small muted"><?= count($sources) ?> total</div>
      </div>

      <div class="tablewrap">
        <table class="table">
          <thead>
            <tr>
              <th style="width:70px">ID</th>
              <th style="width:140px">Type</th>
              <th>Name</th>
              <th style="width:120px">Enabled</th>
              <th style="width:150px">Poll</th>
              <th style="width:170px">Last Polled</th>
              <th style="width:180px">Created</th>
              <th style="width:180px">Updated</th>
              <th style="width:160px">Actions</th>
            </tr>
          </thead>
          <tbody>
            <?php foreach ($sources as $s): ?>
              <?php
                $cfg = json_decode_safe($s['config_json'] ?? '');
                $isEnabled = (int)($s['is_enabled'] ?? 0);

                // Optional "summary" fields from config_json (future-friendly)
                $cfgSummaryParts = [];

                // Common patterns you might store:
                foreach (['base_url','endpoint','account','username','realm_id','store_url','seller_id'] as $k) {
                  if (isset($cfg[$k]) && $cfg[$k] !== '') $cfgSummaryParts[] = "{$k}: " . (string)$cfg[$k];
                }

                // Never show secrets
                foreach (['client_secret','secret','password','api_key','access_token','refresh_token'] as $k) {
                  if (isset($cfg[$k])) {
                    // remove in case you later print $cfg
                    unset($cfg[$k]);
                  }
                }

                $cfgSummary = implode(' • ', array_slice($cfgSummaryParts, 0, 3));
              ?>
              <tr>
                <td class="mono"><?= (int)$s['id'] ?></td>

                <td><span class="pill mono"><?= h($s['type']) ?></span></td>

                <td>
                  <div style="font-weight:900"><?= h($s['name']) ?></div>
                  <?php if ($cfgSummary): ?>
                    <div class="muted" style="font-size:12px"><?= h($cfgSummary) ?></div>
                  <?php endif; ?>
                </td>

                <td>
                  <span class="<?= badge_enabled($isEnabled) ?>">
                    <span class="dot"></span><?= $isEnabled ? 'ENABLED' : 'DISABLED' ?>
                  </span>
                </td>

                <td class="mono">
                  <?= (int)$s['poll_interval_seconds'] ?>s
                </td>

                <td class="muted mono"><?= h(fmt_dt($s['last_polled_at'])) ?: '<span class="muted">(never)</span>' ?></td>
                <td class="muted mono"><?= h(fmt_dt($s['created_at'])) ?></td>
                <td class="muted mono"><?= h(fmt_dt($s['updated_at'])) ?></td>

                <td>
                  <div class="actions">
                    <!-- next: client_view.php / edit -->
                    <a class="btn btn-ghost" style="text-decoration:none" href="client_view.php?id=<?= (int)$s['id'] ?>&tab=overview">
                        View
                    </a>
                  </div>
                </td>
              </tr>
            <?php endforeach; ?>

            <?php if (!count($sources)): ?>
              <tr><td colspan="9" class="muted">No sources found.</td></tr>
            <?php endif; ?>
          </tbody>
        </table>
      </div>

      <!-- Mobile cards -->
      <div class="cards">
        <?php foreach ($sources as $s): ?>
          <?php
            $isEnabled = (int)($s['is_enabled'] ?? 0);
          ?>
          <div class="card">
            <div class="row">
              <div>
                <div class="muted k">Client</div>
                <div class="v"><b><?= h($s['name']) ?></b></div>
              </div>
              <div style="text-align:right">
                <div class="muted k">ID</div>
                <div class="v mono"><?= (int)$s['id'] ?></div>
              </div>
            </div>

            <div class="row">
              <div class="pill mono"><?= h($s['type']) ?></div>
              <span class="<?= badge_enabled($isEnabled) ?>"><span class="dot"></span><?= $isEnabled ? 'ENABLED' : 'DISABLED' ?></span>
            </div>

            <div class="row">
              <div class="muted k">Poll</div>
              <div class="v mono"><?= (int)$s['poll_interval_seconds'] ?>s</div>
            </div>

            <div class="row">
              <div class="muted k">Last polled</div>
              <div class="v mono"><?= h(fmt_dt($s['last_polled_at'])) ?: '(never)' ?></div>
            </div>

            <div class="row">
              <div class="muted k">Updated</div>
              <div class="v mono"><?= h(fmt_dt($s['updated_at'])) ?></div>
            </div>
          </div>
        <?php endforeach; ?>
      </div>

    </div>

    <div style="height:40px"></div>
  </div>
  <?php
}, [
  'active' => 'clients',
]);
