<?php
// admin/client_view.php
// Single source/client details view (tabs)
require $_SERVER['DOCUMENT_ROOT'] . '/net-32-qbo-sync/src/Support/env.php';
$config = require $_SERVER['DOCUMENT_ROOT'] . '/net-32-qbo-sync/config.php';
require $_SERVER['DOCUMENT_ROOT'] . '/net-32-qbo-sync/db.php';
require __DIR__ . '/../layouts/app.php';

date_default_timezone_set($config['timezone'] ?? 'UTC');
$pdo = db($config);

function h($s): string { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

function fmt_dt(?string $dt): string {
  if (!$dt) return '';
  try { return (new DateTime($dt))->format('M j, Y g:i A'); }
  catch (Throwable $e) { return (string)$dt; }
}

function json_decode_safe(?string $s): array {
  if (!$s) return [];
  $j = json_decode($s, true);
  return is_array($j) ? $j : [];
}

function pretty_json_value($value): string {
  $pretty = json_encode($value, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
  return $pretty !== false ? $pretty : '';
}

function redact_config(array $cfg): array {
  // redact common secret keys (including your sample)
  $secretKeys = [
    'sub_key_value','subscription_key','subscriptionKey',
    'api_key','apikey','apiKey',
    'secret','client_secret','clientSecret',
    'password','access_token','refresh_token'
  ];

  foreach ($cfg as $k => $v) {
    $lk = strtolower((string)$k);
    foreach ($secretKeys as $sk) {
      if ($lk === strtolower($sk) || str_contains($lk, 'token') || str_contains($lk, 'secret') || str_contains($lk, 'key')) {
        if (is_string($v) && $v !== '') $cfg[$k] = '***REDACTED***';
      }
    }
  }
  return $cfg;
}

function badge_enabled(int $isEnabled): string {
  return $isEnabled ? 'badge green' : 'badge gray';
}

function badge_class(string $st): string {
  return match ($st) {
    'RECEIVED' => 'badge blue',
    'PROCESSING' => 'badge purple',
    'PROCESSED' => 'badge green',
    'FAILED' => 'badge red',
    'IGNORED' => 'badge gray',
    default => 'badge gray',
  };
}

// Inputs
$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$tab = $_GET['tab'] ?? 'overview';
$tabAllow = ['overview','config','orders','edit'];
if (!in_array($tab, $tabAllow, true)) $tab = 'overview';

if ($id <= 0) {
  header("Location: clients.php?msg=" . urlencode("Missing client id"));
  exit;
}

$msg = $_GET['msg'] ?? '';
$err = $_GET['err'] ?? '';

// Load client
$client = db_one($pdo, "SELECT * FROM sources WHERE id=?", [$id]);
if (!$client) {
  header("Location: clients.php?msg=" . urlencode("Client not found"));
  exit;
}

$cfgRaw = (string)($client['config_json'] ?? '');
$cfgArr = json_decode_safe($cfgRaw);
$cfgRedacted = redact_config($cfgArr);

// Latest orders for this client (optional tab)
$orders = [];
if ($tab === 'orders') {
  $orders = db_all($pdo, "
    SELECT id, external_order_id, status, attempt_count, qbo_txn_id, qbo_txn_type, error_message, updated_at, created_at
    FROM orders
    WHERE source_id=?
    ORDER BY updated_at DESC
    LIMIT 200
  ", [$id]);
}

render_admin_page($config, $pdo, 'Client Details', function() use (
  $config, $id, $tab, $msg,
  $client, $cfgRaw, $cfgArr, $cfgRedacted,
  $orders
) {
  $isEnabled = (int)($client['is_enabled'] ?? 0);
  $type = (string)($client['type'] ?? '');
  $name = (string)($client['name'] ?? '');

  ?>
  <div class="topbar">
    <div class="wrap">
      <div class="title-row">
        <div>
          <h1>Integrations • Client Details</h1>
          <div class="sub">Client #<?= (int)$id ?> • Tabs: Overview / Config / Orders</div>
        </div>
        <div class="pill">
          <span class="dot" style="background: var(--accent)"></span>
          <span class="mono">DB: <?= h($config['db']['name'] ?? '') ?></span>
        </div>
      </div>

      <?php if ($msg): ?>
        <div class="msg"><?= h($msg) ?></div>
      <?php endif; ?>
      <?php if ($err): ?>
        <div class="msg" style="border-color:rgba(239,68,68,.35);background:rgba(239,68,68,.08)"><?= h($err) ?></div>
      <?php endif; ?>
    </div>
  </div>

  <div class="wrap">
    <div class="panel">
      <div class="panel-h">
        <div>
          <div style="display:flex;align-items:center;gap:10px;flex-wrap:wrap">
            <div style="font-weight:900"><?= h($name) ?></div>
            <span class="pill mono"><?= h($type) ?></span>
            <span class="<?= h(badge_enabled($isEnabled)) ?>"><span class="dot"></span><?= $isEnabled ? 'ENABLED' : 'DISABLED' ?></span>
            <span class="pill mono">ID: <?= (int)$client['id'] ?></span>
          </div>
          <div class="small muted">
            Poll: <span class="mono"><?= (int)$client['poll_interval_seconds'] ?>s</span>
            • Last polled: <span class="mono"><?= h(fmt_dt($client['last_polled_at'])) ?: '(never)' ?></span>
            • Updated: <span class="mono"><?= h(fmt_dt($client['updated_at'])) ?></span>
          </div>
        </div>

        <div class="actions">
          <a class="btn btn-ghost" style="text-decoration:none" href="clients.php">Back</a>
          <span class="muted small">(edit/toggle soon)</span>
        </div>
      </div>

      <div class="tabs">
        <a class="tab <?= $tab==='overview'?'active':'' ?>" href="client_view.php?id=<?= (int)$id ?>&tab=overview">Overview</a>
        <a class="tab <?= $tab==='config'?'active':'' ?>" href="client_view.php?id=<?= (int)$id ?>&tab=config">Config</a>
        <a class="tab <?= $tab==='orders'?'active':'' ?>" href="client_view.php?id=<?= (int)$id ?>&tab=orders">Orders</a>
        <a class="tab <?= $tab==='edit'?'active':'' ?>" href="client_view.php?id=<?= (int)$id ?>&tab=edit">Edit</a>
      </div>

      <div class="content">
        <?php if ($tab === 'overview'): ?>
          <div class="grid" style="grid-template-columns: repeat(3, minmax(0,1fr)); margin-top:0">
            <div class="stat">
              <div class="k">Client</div>
              <div class="v" style="font-size:14px;font-weight:900"><?= h($name) ?></div>
              <div class="sub">Type: <span class="mono"><?= h($type) ?></span></div>
              <div class="sub">Enabled: <span class="mono"><?= $isEnabled ? 'yes' : 'no' ?></span></div>
            </div>

            <div class="stat">
              <div class="k">Polling</div>
              <div class="v mono"><?= (int)$client['poll_interval_seconds'] ?>s</div>
              <div class="sub">Last polled: <span class="mono"><?= h(fmt_dt($client['last_polled_at'])) ?: '(never)' ?></span></div>
            </div>

            <div class="stat">
              <div class="k">Timestamps</div>
              <div class="v mono" style="font-size:13px;font-weight:800">
                Created: <?= h(fmt_dt($client['created_at'])) ?>
              </div>
              <div class="sub">Updated: <span class="mono"><?= h(fmt_dt($client['updated_at'])) ?></span></div>
            </div>
          </div>

          <details open>
            <summary>Config summary (redacted)</summary>
            <pre class="json pretty"><?= h(pretty_json_value($cfgRedacted) ?: '(none)') ?></pre>
          </details>

        <?php elseif ($tab === 'config'): ?>
          <details open>
            <summary>Config JSON (redacted)</summary>
            <pre class="json pretty"><?= h(pretty_json_value($cfgRedacted) ?: '(none)') ?></pre>
          </details>

          <details>
            <summary>Raw config_json (unsafe — for debugging)</summary>
            <pre class="json pretty"><?= h($cfgRaw ?: '(none)') ?></pre>
          </details>

        <?php elseif ($tab === 'orders'): ?>
          <div class="small muted" style="margin-bottom:10px">Latest 200 orders for this client</div>

          <div class="tablewrap">
            <table class="table">
              <thead>
                <tr>
                  <th style="width:70px">ID</th>
                  <th>External Order</th>
                  <th>Status</th>
                  <th style="width:120px">Attempts</th>
                  <th style="width:140px">QBO</th>
                  <th>Last Error</th>
                  <th style="width:170px">Updated</th>
                  <th style="width:120px">Actions</th>
                </tr>
              </thead>
              <tbody>
                <?php foreach ($orders as $o): ?>
                  <?php $st = (string)$o['status']; ?>
                  <tr>
                    <td class="mono"><?= (int)$o['id'] ?></td>
                    <td class="mono"><?= h($o['external_order_id']) ?></td>
                    <td><span class="<?= h(badge_class($st)) ?>"><span class="dot"></span><?= h($st) ?></span></td>
                    <td class="mono"><?= (int)$o['attempt_count'] ?></td>
                    <td>
                      <?php if (!empty($o['qbo_txn_id'])): ?>
                        <div style="font-weight:800"><?= h($o['qbo_txn_type']) ?></div>
                        <div class="mono"><?= h($o['qbo_txn_id']) ?></div>
                      <?php else: ?>
                        <span class="muted">(none)</span>
                      <?php endif; ?>
                    </td>
                    <td class="muted" style="max-width:420px"><?= h((string)($o['error_message'] ?? '')) ?></td>
                    <td class="muted mono"><?= h(fmt_dt($o['updated_at'])) ?></td>
                    <td>
                      <a class="btn btn-ghost" style="text-decoration:none" href="order_view.php?id=<?= (int)$o['id'] ?>&tab=overview">View</a>
                    </td>
                  </tr>
                <?php endforeach; ?>
                <?php if (!count($orders)): ?>
                  <tr><td colspan="8" class="muted">No orders for this client.</td></tr>
                <?php endif; ?>
              </tbody>
            </table>
          </div>
        <?php elseif ($tab === 'edit'): ?>
            <form method="POST" action="/net-32-qbo-sync/admin/actions/client_update.php" style="max-width:980px">
                <input type="hidden" name="id" value="<?= (int)$client['id'] ?>">

                <div class="grid" style="grid-template-columns: repeat(2, minmax(0,1fr)); margin-top:0">
                <div class="stat">
                    <div class="k">Type</div>
                    <div class="sub">Type is read-only (for now)</div>
                    <input type="text" value="<?= h($type) ?>" disabled>
                </div>

                <div class="stat">
                    <div class="k">Name</div>
                    <div class="sub">Friendly label</div>
                    <input type="text" name="name" value="<?= h($name) ?>" required>
                </div>

                <div class="stat">
                    <div class="k">Enabled</div>
                    <div class="sub">Disable to pause polling/processing</div>
                    <select name="is_enabled">
                    <option value="1" <?= $isEnabled ? 'selected' : '' ?>>Enabled</option>
                    <option value="0" <?= !$isEnabled ? 'selected' : '' ?>>Disabled</option>
                    </select>
                </div>

                <div class="stat">
                    <div class="k">Poll Interval (seconds)</div>
                    <div class="sub">Minimum 30 seconds</div>
                    <input type="number" name="poll_interval_seconds" min="30" step="1"
                        value="<?= (int)$client['poll_interval_seconds'] ?>" required>
                </div>
                </div>

                <div class="stat" style="margin-top:14px">
                <div class="k">Config JSON</div>
                <div class="sub">Must be valid JSON. Secrets are stored as-is.</div>
                <textarea name="config_json" rows="14"
                    style="width:100%; font-family: ui-monospace, SFMono-Regular, Menlo, monospace; border-radius:12px; padding:12px; border:1px solid rgba(255,255,255,.12); background:rgba(255,255,255,.02); color:inherit;"><?= h(pretty_json_value($cfgArr) ?: $cfgRaw) ?></textarea>
                </div>

                <div style="margin-top:14px; display:flex; gap:10px; align-items:center;">
                <button class="btn" type="submit">Save Changes</button>
                <a class="btn btn-ghost" style="text-decoration:none"
                    href="client_view.php?id=<?= (int)$id ?>&tab=overview">Cancel</a>
                </div>

                <div class="muted" style="margin-top:10px; font-size:12px;">
                Tip: if JSON validation fails, you’ll be redirected back with an error message.
                </div>
            </form>
        <?php endif; ?>
      </div>

    <div style="height:40px"></div>
  </div>
  <?php
}, [
  'active' => 'clients',
]);
