<?php
// lib/qbo/invoice.php

function qbo_build_invoice_payload(
  array $config,
  PDO $pdo,
  string $token,
  int $orderId,
  array $normalized,
  string $qboCustomerId
): array {

  $defaultItemId = (string)($config['qbo']['default_item_id'] ?? '');
  if ($defaultItemId === '') {
    throw new Exception("Missing qbo.default_item_id");
  }

  $lines = [];

  // Product/Services Sales lines
  foreach (($normalized['lines'] ?? []) as $li) {
    $qty = (float)($li['qty'] ?? 1);
    if ($qty <= 0) $qty = 1;

    $unit   = money((float)($li['unit_price'] ?? 0));
    $amount = money($unit * $qty);

    // ✅ Try SKU mapping first
    $sku = trim((string)($li['sku'] ?? $li['vpCode'] ?? ''));
    $itemId = $defaultItemId;

    log_line($config, "Processing line with SKU '{$sku}' qty={$qty} unit={$unit} amount={$amount}");
    order_event($pdo, $orderId, 'QBO_PRODUCT_LOOKUP', [
        'SKU' => (string)$sku,
        'qty' => (string)$qty,
        'unit' => (string)$unit,
        'amount' => (string)$amount,
    ]);

    if ($sku !== '') {
      $item = qbo_find_item_by_sku($config, $pdo, $token, $orderId, $sku);
      if (is_array($item) && !empty($item['Id'])) {
        $itemId = (string)$item['Id'];
        order_event($pdo, $orderId, 'QBO_PRODUCT_FOUND', [
            'qbo_item_id' => $itemId,   
            'SKU' => (string)$sku,
            'qty' => (string)$qty,
            'unit' => (string)$unit,
            'amount' => (string)$amount,
        ]);
      } else {
        order_event($pdo, $orderId, 'QBO_PRODUCT_NOT_FOUND', [
            'SKU' => (string)$sku,
            'qty' => (string)$qty,
            'unit' => (string)$unit,
            'amount' => (string)$amount,
        ]);
      }
    }

    $lines[] = [
      'Amount' => $amount,
      'DetailType' => 'SalesItemLineDetail',
      'Description' => (string)($li['description'] ?? 'Item'),
      'SalesItemLineDetail' => [
        'ItemRef' => ['value' => $itemId],
        'Qty' => $qty,
        'UnitPrice' => $unit,
      ],
    ];
  }

  // Shipping line
  $shipping = money((float)($normalized['totals']['shipping'] ?? 0));
  if ($shipping > 0) {
    $shipItemId = (string)($config['qbo']['shipping_item_id'] ?? '');
    if ($shipItemId === '') $shipItemId = $defaultItemId;

    $lines[] = qbo_simple_item_line($shipping, 'Shipping', $shipItemId);
  }

  // Tax line
  $tax = money((float)($normalized['totals']['tax'] ?? 0));
  if ($tax > 0 && $config['qbo']['taxes_included_in_amount'] === true) {
    $taxItemId = (string)($config['qbo']['tax_item_id'] ?? '');
    if ($taxItemId === '') $taxItemId = $defaultItemId;

    $lines[] = qbo_simple_item_line($tax, 'Tax', $taxItemId);
  }

  $po = trim((string)($normalized['external_order_id'] ?? ''));

  $payload = [
    'CustomerRef' => ['value' => $qboCustomerId],
    'Line' => $lines,
    'PrivateNote' =>
      'OrderId: ' . ($normalized['external_order_id'] ?? '') .
      ' | Source: ' . ($normalized['source'] ?? ''),
  ];

  if ($po !== '') {
    $payload['PONumber'] = $po;
  }

  // Custom fields
  $customFields = [];

  $poDefId = (string)($config['qbo']['custom_fields']['po_number_definition_id'] ?? '');
  if ($po !== '' && $poDefId !== '') {
    $customFields[] = [
      'DefinitionId' => $poDefId,
      'Type' => 'StringType',
      'StringValue' => $po,
    ];
  }

  $licDefId = (string)($config['qbo']['custom_fields']['license_number_definition_id'] ?? '');
  if ($licDefId !== '' && !empty($normalized['license']) && is_array($normalized['license'])) {
    $licNum  = trim((string)($normalized['license']['license_number'] ?? ''));
    $licType = trim((string)($normalized['license']['license_type'] ?? ''));

    $licenseValue = '';
    if ($licType !== '' && $licNum !== '') $licenseValue = "{$licType} #: {$licNum}";
    elseif ($licNum !== '') $licenseValue = $licNum;
    elseif ($licType !== '') $licenseValue = $licType;

    if ($licenseValue !== '') {
      $customFields[] = [
        'DefinitionId' => $licDefId,
        'Type' => 'StringType',
        'StringValue' => $licenseValue,
      ];
    }
  }

  if ($customFields) {
    $payload['CustomField'] = $customFields;
  }

  // Ship To
  if (!empty($normalized['ship_to']) && is_array($normalized['ship_to'])) {
    $addr = qbo_build_address($normalized['ship_to']);
    if ($addr) $payload['ShipAddr'] = $addr;
  }

  // Ship From
  if (!empty($config['qbo']['ship_from']) && is_array($config['qbo']['ship_from'])) {
    $addr = qbo_build_address($config['qbo']['ship_from']);
    if ($addr) $payload['ShipFromAddr'] = $addr;
  }

  // Bill To
  if (!empty($normalized['bill_to']) && is_array($normalized['bill_to'])) {
    $addr = qbo_build_address($normalized['bill_to']);
    if ($addr) $payload['BillAddr'] = $addr;
  }

  return $payload;
}

function qbo_simple_item_line(float $amount, string $desc, string $itemId): array {
  $amount = money($amount);
  $itemId = (string)$itemId;

  return [
    'Amount' => $amount,
    'DetailType' => 'SalesItemLineDetail',
    'Description' => $desc,
    'SalesItemLineDetail' => [
      'ItemRef' => ['value' => $itemId],
      'Qty' => 1,
      'UnitPrice' => $amount,
    ],
  ];
}
