<?php
// lib/db/orders.php

function orders_fetch_eligible(PDO $pdo, int $batchSize, int $maxAttempts): array {
  // NOTE: PDO does not always allow binding LIMIT as parameter depending on driver settings.
  // We'll hard cast and inject safe integer.
  $batchSize = max(1, (int)$batchSize);

  $sql = "
    SELECT o.*, s.type AS source_type, s.name AS source_name
    FROM orders o
    JOIN sources s ON s.id = o.source_id
    WHERE o.status IN ('RECEIVED','FAILED')
      AND (o.next_retry_at IS NULL OR o.next_retry_at <= NOW())
      AND o.attempt_count < :max_attempts
      AND s.is_enabled = 1
    ORDER BY o.updated_at ASC
    LIMIT {$batchSize}
  ";

  $st = $pdo->prepare($sql);
  $st->execute([':max_attempts' => $maxAttempts]);
  return $st->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Atomically claim an order row for processing, increments attempt_count.
 * Returns true if claim succeeded, false if not eligible anymore.
 */
function orders_claim(PDO $pdo, int $orderId, int $maxAttempts): bool {
  $n = db_exec($pdo, "
    UPDATE orders
    SET status='PROCESSING',
        attempt_count = attempt_count + 1,
        last_attempt_at = NOW(),
        updated_at = NOW()
    WHERE id = :id
      AND status IN ('RECEIVED','FAILED')
      AND (next_retry_at IS NULL OR next_retry_at <= NOW())
      AND attempt_count < :max_attempts
  ", [
    ':id' => $orderId,
    ':max_attempts' => $maxAttempts,
  ]);

  return $n === 1;
}

function orders_store_normalized(PDO $pdo, int $orderId, array $normalized): void {
  db_exec($pdo, "UPDATE orders SET normalized_json=:norm, updated_at=NOW() WHERE id=:id", [
    ':norm' => json_encode($normalized, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT),
    ':id' => $orderId
  ]);
}

function orders_mark_processed(PDO $pdo, int $orderId, string $txnType, ?string $txnId, ?string $customerId): void {
  db_exec($pdo, "
    UPDATE orders
    SET status='PROCESSED',
        qbo_txn_type=:t,
        qbo_txn_id=:txn,
        qbo_customer_id=:cust,
        processed_at=NOW(),
        error_code=NULL,
        error_message=NULL,
        next_retry_at=NULL,
        updated_at=NOW()
    WHERE id=:id
  ", [
    ':t' => $txnType,
    ':txn' => $txnId,
    ':cust' => $customerId,
    ':id' => $orderId,
  ]);
}

function orders_mark_failed(PDO $pdo, int $orderId, string $message, ?string $nextRetryAt): void {
  db_exec($pdo, "
    UPDATE orders
    SET status='FAILED',
        error_code=NULL,
        error_message=:msg,
        next_retry_at=:next_retry,
        updated_at=NOW()
    WHERE id=:id
  ", [
    ':msg' => $message,
    ':next_retry' => $nextRetryAt,
    ':id' => $orderId,
  ]);
}

function orders_get_attempt_count(PDO $pdo, int $orderId): int {
  $row = db_one($pdo, "SELECT attempt_count FROM orders WHERE id=?", [$orderId]);
  return (int)($row['attempt_count'] ?? 0);
}
